# migration script v1.20
# Action1 Corporation holds no liability for any damages directly or indirectly caused by running this script.

# Before migrating enpoints, please make sure:
# - the enpoints are not managed by Action1 Deployer;
# - no one policy is running against them, also recommended to disable policies;
# - the following parameter "App Name Match" is set to ^(Action1AgentMigration)$ in the Web User Interface;
# - the following parameter "Version Number" is set to 1.20 in the Web User Interface;
# - the following parameter "Silent install switches" is set to launcher.ps1 in the Web User Interface;
# - the script is successfully completed on one endpoint.


Function Main {
    $Script:serviceName = 'A1Agent';
    $Script:agentPath = "$env:windir\Action1";
    $action1KeyPath64 = "HKLM:\SOFTWARE\WOW6432Node\Action1";
    $action1KeyPath32 = "HKLM:\SOFTWARE\Action1";
    $Script:prefix = "old_";
    $logname = "migration.log";
    $Script:LogFile = "$agentPath\logs\$logname";

    $Script:TempKeyPath = 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall';
    $Script:TempKeyName = $Script:TempDisplayName = 'Action1AgentMigration';
    $Script:TempDisplayVersion = '1.20';

    $msifile = '.\*.msi';
        
    $Script:includeForRemove = "action1_agent.exe.connection", "action1_agent.exe.instance_lock", "action1_agent.exe.update_state",
    "advanced_settings.json", "agent_downgrade.version_pair", "rules.json",
    "schedule.json", "alert_queue", "batch_data", "datasource_data",
    "package_downloads", "rule_data", "running_batches", "scripts";
    
    if (-not(Test-Path -Path "${agentPath}\logs")) {
        try {
            New-Item -ItemType Directory -Force -Path "${agentPath}\logs" -ErrorAction Stop;
        } catch { exit 1; }
    }

    $msifile = Get-ChildItem -Path $msifile -ErrorAction SilentlyContinue;
    
    if ($msifile -eq $null) {
        Log -Message "There is no MSI file in the archive. " -ErrorLevel "ERR";
        exit 1;
    }
    elseif ($msifile.GetType().BaseType.ToString() -eq "System.Array") {
        Log -Message "There are found multiple MSI files in the archive." -ErrorLevel "ERR";
        exit 1;
    }

    Log -Message "The Action1 agent migration is started.";

    if (-not(Test-Path -Path "${agentPath}\action1_agent.exe" -PathType Leaf)) {
        $message = '"{0}" does not exist.' -f "${agentPath}\action1_agent.exe";
        Log -Message "$message" -ErrorLevel "ERR";
        exit 1;
    }

    if (Test-Path -Path "${action1KeyPath64}") {
        $Script:KeyPath = $action1KeyPath64;
        Change-Organization;
        exit 0;
    }

    if (Test-Path -Path "${action1KeyPath32}") {
        $Script:KeyPath = $action1KeyPath32;
        Change-Organization;
        exit 0;
    }

    Log -Message 'The registry key "Action1" is not found.' -ErrorLevel "ERR";
    exit 1;
}


Function Extract-MsiProperty {
    Param ($PropName, [string]$MSIfile)
    $null = $WindowsInstaller = $database = $view = $null;

    try {
        $windowsInstaller = New-Object -ComObject WindowsInstaller.Installer;
        $database = $windowsInstaller.GetType().InvokeMember("OpenDatabase", "InvokeMethod", $null, $windowsInstaller, @("$MSIfile", 0));
        $query = "SELECT * FROM Property WHERE Property = '{0}' " -f $PropName;
        $view = $database.GetType().InvokeMember("OpenView", "InvokeMethod", $null, $database, ("$query"));
        $view.GetType().InvokeMember("Execute", "InvokeMethod", $null, $view, $null) | Out-Null;
        
        $row = $view.GetType().InvokeMember("Fetch", "InvokeMethod", $null, $view, $null);
        if ($row) { $propertyValue = $row.GetType().InvokeMember("StringData", "GetProperty", $null, $row, 2); }
        $view.GetType().InvokeMember("Close", "InvokeMethod", $null, $view, $null) | Out-Null;
    }
    catch {
        Log -Message $_.Exception.Message -ErrorLevel "ERR";
        exit 1;
    }

    $null = [System.Runtime.InteropServices.Marshal]::FinalReleaseComObject($windowsInstaller);
    $null = [System.Runtime.InteropServices.Marshal]::FinalReleaseComObject($database);
    $null = [System.Runtime.InteropServices.Marshal]::FinalReleaseComObject($view);
    return $propertyValue;
}


Function Log {
    Param ([string]$Message, [string]$ErrorLevel)
    #Errorlevels: INFO, WARN, ERR, HIST
    if (!$ErrorLevel) { $ErrorLevel = "INFO"; }
    $callerFuncName = $(Get-PSCallStack)[1].Command;
    $outputStr = $((Get-Date).ToString("yyyy-MM-dd HH:mm:ss (local): ") + "[" + $callerFuncName + "]: " + $ErrorLevel + ": " + $Message);
    Add-Content -Path $LogFile -Value $outputStr -Force -ErrorAction SilentlyContinue;
}


Function Create-TempRegkey {
    try {
        New-Item -Path "Registry::$TempKeyPath" -Name "$TempKeyName" -Force -ErrorAction Stop | Out-Null;
        New-ItemProperty -Path "Registry::$TempKeyPath\$TempKeyName" -Name "DisplayName" -Value "$TempDisplayName" -PropertyType string -Force -ErrorAction Stop | Out-Null;
        New-ItemProperty -Path "Registry::$TempKeyPath\$TempKeyName" -Name "DisplayVersion" -Value "$TempDisplayVersion" -PropertyType string  -Force -ErrorAction Stop | Out-Null;
        Log -Message $('The temporary registry key "{0}" is created.' -f $("$TempKeyPath\$TempKeyName"));
        [int]$sleepSec = 120;
        Log -Message $('Waiting for {0} seconds before migrating to the destination organization.' -f $sleepSec);
        Start-Sleep -Seconds $sleepSec;
        Log -Message "The migration to the destination organization is started.";
    } catch {
        $message = 'Failed to create the temporary registry key "{0}".' -f $("$TempKeyPath\$TempKeyName");
        Log -Message "$message" -ErrorLevel "ERR";
    }
}


Function Remove-TempRegkey {
    try {
        Remove-Item -Path "Registry::$TempKeyPath\$TempKeyName" -Force -ErrorAction Stop | Out-Null;
        Log -Message $('The temporary registry key "{0}"is removed.' -f $("$TempKeyPath\$TempKeyName"));
    } catch {
        $message = 'Failed to remove the temporary registry key "{0}".' -f $("$TempKeyPath\$TempKeyName");
        Log -Message "$message" -ErrorLevel "ERR";
    }
}


Function Stop-Action1Service {
    $processName = "action1_agent";

    $timeout = 15;
    $service = Get-Service -Name $serviceName -ErrorAction SilentlyContinue; 
    if ($service.Length -eq 0) {
        Log -Message 'The Action1 agent service is not installed.' -ErrorLevel "ERR";
        exit 1;
    }     
    Stop-Service -Name $serviceName | Out-Null;

    while ($service.Status -ne "Stopped") {
        if ($timeout -eq 0) {
            break;
        }    
        start-sleep -seconds 1;
        $timeout--;
        $service.Refresh();
    }

    $timeout = 120;
    $process = Get-Process -Name $processName -ErrorAction SilentlyContinue;
    while ($process) {
        Stop-Process -Name $processName -Force -Confirm:$false;
        if ($timeout -eq 0) {
            break;
        }    
        start-sleep -seconds 1;
        $timeout--;
        $process = Get-Process -Name $processName -ErrorAction SilentlyContinue;
    }
}


Function Start-Action1Service {
    $timeout = 120
    $service = Get-Service -Name $serviceName -ErrorAction SilentlyContinue; 

    Start-Service -Name $serviceName | Out-Null;
    while ($service.Status -ne "Running") {
        if ($timeout -eq 0) {
            Log -Message "The Action1 service is not running for ${timeout} seconds." -ErrorLevel "ERR";
            break;
        }    
        start-sleep -seconds 1;
        $timeout--;
        $service.Refresh();
    }
    Log -Message 'The Action1 service is successfully started.';
}


Function Copy-Action1Property {
    Param ( 
        [string] $KeyPath,
        [string] $PropertyName )

    try {
        $property = Get-ItemProperty -Path "$KeyPath";
        $propertyValue = $property."${PropertyName}";
        if (!$property."${prefix}${PropertyName}") {
            New-ItemProperty -Path "$KeyPath" -Name "${prefix}${PropertyName}" -PropertyType String -Value "$propertyValue" -Force -ErrorAction Stop | Out-Null;
            Log -Message $('The value "{1}{2}" of the registry key "{0}" is created successfully.' -f $KeyPath, $prefix, $PropertyName);
        }
        else {
            Log -Message $('The value "{1}{2}" of the registry key "{0}" is already exist.' -f $KeyPath, $prefix, $PropertyName);
        }    
    }
    catch {
        Log -Message $_.Exception.Message -ErrorLevel "ERR";
    }
}


Function Change-Organization {

    if (-not(Test-Path -Path "${KeyPath}")) {
        $message = 'The registry key "{0}" does not exist.' -f $KeyPath;
        Log -Message "$message" -ErrorLevel "ERR";
        exit 1;
    }

    if (-not(Test-Path -Path "${KeyPath}\Agent")) {
        $message = 'The registry key "{0}" does not exist.' -f "${KeyPath}\Agent"
        Log -Message "$message"  -ErrorLevel "ERR";
        exit 1;
    }
    
    try {
        $customerId = Extract-MsiProperty -PropName 'A1_CUSTOMERID' -MSIfile $msifile;
        $privateKey = Extract-MsiProperty -PropName 'A1_PRIVATEKEY' -MSIfile $msifile;
        $certificate = Extract-MsiProperty -PropName 'A1_CERTIFICATE' -MSIfile $msifile;

        $property = Get-ItemProperty -Path "${KeyPath}\Agent";
        $propertyValue = $property."CustomerId";
    
        if ($customerId -eq $propertyValue) {
            $message = "The source and destination organization are the same.";
            Log -Message "$message" -ErrorLevel "ERR";
            exit 1;      
        }
        
        Create-TempRegkey;
        Stop-Action1Service;
        Rename-Action1Items | Out-Null;
        Copy-RegistyItems | Out-Null; 
    
        $uninstallMsi = Start-Process "${agentPath}\action1_agent.exe" -ArgumentList "uninstall-msi" -PassThru -Wait -ErrorAction Stop;
        if (-not $uninstallMsi) {
            $message = "The ""null"" is returned while unregistering the agent.";
            throw "$message";
        }
    
        if ($uninstallMsi.ExitCode -ne 0) {
            $message = "The following code is returned ""$($uninstallMsi.ExitCode)"" while unregistering the agent.";
            throw "$message";
        }
        
        New-ItemProperty -Path "${KeyPath}" -Name 'agent.guid' -PropertyType String -Value "" -Force -ErrorAction Stop | Out-Null;
        New-ItemProperty -Path "${KeyPath}\Agent" -Name 'CustomerId' -PropertyType String -Value "$customerId" -Force -ErrorAction Stop | Out-Null;
        New-ItemProperty -Path "${KeyPath}\Agent" -Name 'PrivateKey' -PropertyType String -Value "$privateKey" -Force -ErrorAction Stop | Out-Null;
        New-ItemProperty -Path "${KeyPath}\Agent" -Name 'Certificate' -PropertyType String -Value "$certificate" -Force -ErrorAction Stop | Out-Null;
    }
    catch {
        Log -Message $_.Exception.Message -ErrorLevel "ERR";
        Rollback-Changes | Out-Null;
        
    }

    Remove-TempRegkey;
    Start-Action1Service;
    Log -Message "The agent migration is completed.";
}


Function Copy-RegistyItems {
    Copy-Action1Property -KeyPath "${KeyPath}" -PropertyName "agent.guid";
    Copy-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "Certificate";
    Copy-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "CustomerId";
    Copy-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "PrivateKey";
}


Function Rename-Action1Items {
    param ([bool]$Rollback = $false)

    $oldPrefix = "";
    $newPrefix = $Prefix;
    if ($Rollback) {
        $oldPrefix = "$Prefix";
        $newPrefix = "";
    }

    foreach ($item in $includeForRemove) {
        $oldName = Join-Path -Path "$agentPath" -ChildPath $("$OldPrefix" + "$item");
        $newName = Join-Path -Path "$agentPath" -ChildPath $("$newPrefix" + "$item");
        try {
            if (-not(Test-Path -Path "$oldName")) {
                Log -Message """$oldName"" does not exist.";
                continue;
            }
            $isRenamed = Rename-Item -Path "$oldName" -NewName "$newName" -Force -PassThru -ErrorAction SilentlyContinue;
            if (-not $isRenamed) {
                Start-Sleep -Seconds 2
                Rename-Item -Path "$oldName" -NewName "$newName" -Force -ErrorAction Stop;
            }
            Log -Message "The item ""$oldName"" is renamed to ""$newName"".";
        }
        catch {
            Log -Message "Failed to rename ""$oldName"" to ""$newName"". $($_.Exception.Message)" -ErrorLevel "WARN";
        }
    }
}


Function Rollback-Changes {
    Rollback-Registry;
    Rename-Action1Items -Rollback $true;
}


Function Rollback-Registry {
    Rollback-Action1Property -KeyPath "${KeyPath}" -PropertyName "agent.guid";
    Rollback-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "Certificate";
    Rollback-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "CustomerId";
    Rollback-Action1Property -KeyPath "${KeyPath}\Agent" -PropertyName "PrivateKey";
}


Function Rollback-Action1Property {
    param ( [string] $KeyPath,
            [string] $PropertyName )
    
    try {
        Remove-ItemProperty -Path "$KeyPath" -Name ${prefix}${PropertyName} -Force -ErrorAction Stop | Out-Null;
        Log -Message $('The value "{1}{2}" of the registry key "{0}" is removed successfully.' -f $KeyPath, $Prefix, $PropertyName);
    }
    catch {
        $message = 'Failed to rollback the value "{1}{2}" of the registry key "{0}". Exception: {3}' -f $KeyPath, $Prefix, $PropertyName, "$($_.Exception.Message)";
        Log -Message $message -ErrorLevel "ERR";
    }
}


Main;
